/*---------------------------------------------------------------------------*\

	FILE....: kldrop.c
	TYPE....: C Function
	AUTHOR..: David Rowe
	DATE....: 16/8/02
	AUTHOR..: Ben Kramer
	DATE....: 29/1/03

	Kernel based Loop Drop detection module for V12PCI, links with hda.c
	Based on the kring.c module and codec.c module

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Voicetronix Voice Processing Board (VPB) Software

	Copyright (C) 1999-2002 Voicetronix www.voicetronix.com.au

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	USA

\*---------------------------------------------------------------------------*/

//#include <linux/kernel.h>
//#include <asm-i386/spinlock.h>

//#include <linux/autoconf.h>
//#include <linux/modversions.h>


//#include "vpb_ioctl.h"
//#include "fifoc.h"

#include "kldrop.h"

/*--------------------------------------------------------------------------*\

DEFINES

\*--------------------------------------------------------------------------*/

#define V12PCI_PORTS    12    /* number of ports per V12PCI             */
#define	WINDOW          200   /* no of SPERIODS to sample (200ms)       */
#define THRESH_RING     60    /* no. of H samples in window for RING    */
#define THRESH_SIL      0     /* no. of H samples in window for SIL     */

/* loop drop detector states */

#define DFRONT_L        0     /* front porch (L) for DWINL samples      */
#define DWAITH_WIN      1     /* wait while H for DWINH                 */
#define DBACK_L         2     /* back porch (L) for DWINL               */

/* loop drop detector constants */

#define DWINL           90    /* 90ms */
#define DWINH           50    /* 50ms */


/*--------------------------------------------------------------------------*\

STATICS

\*--------------------------------------------------------------------------*/

typedef struct {
	unsigned short *base2;                    // V12PCI card memory
	int            ldrop_state[V12PCI_PORTS];  // ring det state   
	int            count[V12PCI_PORTS];       
	int            ldrop_evt[V12PCI_PORTS];    // asserted for ld event
} KLDROP, *PKLDROP; 

// maps port to addr, bit pairs for ring det for that port

int ldropmap[] = {
	7,0,
	7,2,
	7,4,
	7,6,
	8,0,
	8,2,
	8,4,
	8,6,
	9,0,
	9,2,
	9,4,
	9,6
};

/*--------------------------------------------------------------------------*\

	FUNCTION.: kring_open()
	AUTHOR...: David Rowe
	DATE.....: 1/8/02
	AUTHOR..: Ben Kramer
	DATE....: 29/1/03

	Constructor to init hook detection module for a board.

\*--------------------------------------------------------------------------*/

void *kldrop_open(unsigned short *base2) {
	PKLDROP kldrop;
	int    ch;

	kldrop = kldrop_malloc(sizeof(KLDROP));

	for(ch=0; ch<V12PCI_PORTS; ch++) {
		kldrop->ldrop_state[ch] = DFRONT_L;
		kldrop->ldrop_evt[ch] = 0;
	}
	kldrop->base2 = base2;

	return((void*)kldrop);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: kring_close()
	AUTHOR...: David Rowe
	DATE.....: 17/8/02
	AUTHOR..: Ben Kramer
	DATE....: 29/1/03

	Destructor to close down ring detection module.

\*--------------------------------------------------------------------------*/

void kldrop_close(void *kldrop) {
	kldrop_free(kldrop);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: kldrop_sample
	AUTHOR...: David Rowe
	DATE.....: 16/8/02
	AUTHOR..: Ben Kramer
	DATE....: 29/1/03

	Called from ISR every 1ms to sample ring bits and iterate ld det
	state machine.

\*--------------------------------------------------------------------------*/

void kldrop_sample(void *pv_kldrop, int btb)
{
	PKLDROP  kldrop = (PKLDROP)pv_kldrop;
	int     ring, state, next_state, addr, bit;
	int     ch;
	int     *ldrop_state = kldrop->ldrop_state;
	int     *count = kldrop->count;
	int     *ldrop_evt = kldrop->ldrop_evt;
	
	for(ch=btb; ch<V12PCI_PORTS; ch++) {

		// sense ring bit in i-th channel 

		addr = ldropmap[ch*2];
		bit = 1 << ldropmap[ch*2+1];
		ring = kldrop_readw(kldrop->base2 + addr);
		ring &= bit;

		state = ldrop_state[ch];

		next_state = state;
		switch(state) {

		case DFRONT_L:
			// Wait for DWINL L samples  
			if (count[ch]++ < DWINL) {
				if (ring) {
					// if H reset 
					count[ch] = 0;
					next_state = DFRONT_L;
				}
			}
			else {
			// now wait for spike (single H sample) 
				if (ring) {
					count[ch] = 0;
					next_state = DWAITH_WIN;
//					printk("<7> Change to DWAITH_WIN\n");
				}
				else {
					next_state = DFRONT_L;
				}
			}
		break;
		case DWAITH_WIN:
			// Wait for DWINH samples  
			if (count[ch]++ > DWINH) {
				count[ch] = 0;
				next_state = DBACK_L;
				//printk("<7> Change to DBACK_L\n");
			}
			else
				next_state = DWAITH_WIN;
		break;

		case DBACK_L:
			// Wait for DWINL L samples  
			if (count[ch]++ < DWINL) {	    
				if (ring) {
					// if H reset 
					count[ch] = 0;
					next_state = DFRONT_L;
				//printk("<7> Change to DFRONT_L\n");
				}
			}
			else {
				// generate event
				//printk("<7> Hey Dude, got a Loop Drop on channel %d !\n", ch);
				ldrop_evt[ch] = 1;
				count[ch] = 0;
				next_state = DFRONT_L;
			}
		break;

		default:
			next_state = state;

		}

		ldrop_state[ch] = next_state;

	} // for(ch=0 ... 
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: kring_read()
	AUTHOR...: David Rowe
	DATE.....: 18/8/02
	AUTHOR..: Ben Kramer
	DATE....: 29/1/03

	Called by driver IOCTL to determine if a ring event has occurred.
	After reading, the ring_evt flag is reset.

\*--------------------------------------------------------------------------*/

int kldrop_read(void *pv_kldrop, int ch) {
	PKLDROP  kldrop = (PKLDROP)pv_kldrop;
	int     *ldrop_evt = kldrop->ldrop_evt;
	int     ret;

	ret =  ldrop_evt[ch];
	ldrop_evt[ch] = 0;
	return ret;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: kring_getevt()
	AUTHOR..: Ben Kramer
	DATE....: 30/1/03

	Used to check the event flags for all channels.

\*--------------------------------------------------------------------------*/

int kldrop_getevt(void *pv_kldrop) {
	PKLDROP  kldrop = (PKLDROP)pv_kldrop;
	int     *ldrop_evt = kldrop->ldrop_evt;
	int     ret,ch;

	for (ch=0;ch < V12PCI_PORTS;ch++){
		ret +=  ldrop_evt[ch];
	}

	return ret;
}

